"""Analysis utilities for Wilson-loop measurements.

This module defines a single function ``fit_string_tension`` which
performs a linear regression on the logarithm of Wilson-loop
expectation values to extract the area‑law string tension and an
associated 95% confidence interval.  It is a simplified version of
the fitting routine used in the full pipeline and operates on a
dictionary of loop sizes to lists of complex Wilson-loop values.
"""

from typing import Dict, List, Tuple
import numpy as np


def fit_string_tension(loops: Dict[int, List[complex]]) -> Tuple[float, float]:
    """Fit the string tension from Wilson-loop data via linear regression.

    Parameters
    ----------
    loops : dict
        Mapping loop size ``L`` to a list of Wilson-loop values.  The
        magnitude of the average value is used for the fit.

    Returns
    -------
    sigma : float
        Estimated string tension.  This is minus the slope of
        ``log(|W|)`` versus the loop area ``L**2``.
    ci95 : float
        Half-width of the 95% confidence interval for the fitted slope.
    """
    # Sort loop sizes to ensure consistent ordering
    Ls = np.array(sorted(loops.keys()))
    # Compute the average Wilson-loop value for each size
    W_avgs = np.array([np.mean(loops[L]) for L in Ls])
    # Compute areas (L^2)
    As = Ls ** 2
    # Take the logarithm of the magnitude of the averages
    logW = np.log(np.abs(W_avgs))
    # Perform linear regression logW = m * As + c
    m, c = np.polyfit(As, logW, 1)
    # Compute residuals and the standard error of the slope
    y_pred = m * As + c
    resid = logW - y_pred
    n = len(As)
    # Standard error of the slope m
    se_m = np.sqrt(np.sum(resid**2) / (n - 2) / np.sum((As - np.mean(As))**2))
    sigma = -m
    ci95 = 1.96 * se_m
    return sigma, ci95